/*
IRCStats 1.2 - Copyright (c) 1999-2013 Teemu Toivola <tst at iki dot fi>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2 dated June, 1991.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program;  if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
*/


/*

Also see the "About the source" chapter in the README.

*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <string.h>

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__) || defined(__APPLE__)
#include <sys/time.h>
#endif

#define MAXLINE 512 /* length of lines */
#define NICKMAX 30 /* length of nicks */
#define CONFIGID 10 /* id for the config */
#define SMALLTOP 60 /* number of nicks show in the small top */
#define SORTFREQ 5000 /* after how many lines the nick array should be sorted */
#define NICKLIMIT 20000 /* how many different nicks can there be */

#define IVERSION "1.2"
#define IYEARS "1999-2013"

typedef struct {
	char nick[NICKMAX];
	int talk[4];
	char quote[512];
	int join;
	float linelength;
	int yell;
	int ques;
	int mono1;
	int mono2;	/* actual monologue-counter */
	int ttalk;
	int place;	/* not used currently */
	int dummy;
	int action;
	int kick;
	int kicked;
	int opped;
	int deopped;
	int happy;
	int sad;
	/* for future extensions */
	int foo1;
	char foo4[512];
	char foo5[512];
} USER;

typedef struct {
	int h[24];
	int day;
	int quotenbr;
	long totalLines;
	char topic[5][256];
	int topickayt;
	long currentLine;
	int dayact[31][4];
	int monthact[12];
	char lastline[512];	/* 	this is the last line of the log
						to be used if the history system tries to continue */
	int minorver;
	/* for future extensions */
	int foo2;
	int foo3;
	char foo4[512];
	char foo5[512];
} SETUP;

typedef struct {
	int version;
	int dummy;
} SETUPVERSION;

/* Funktiot */
void countLines(void);
int seekline(long line);
void printDots(long linecounter);
void parseTime(void);
void startHtml(void);
void timetable(void);
void parseAction(int offset);
void parseChatline(int offset);
void printNicks(void);
void longEnd(void);
void shortEnd(void);
void loadConfig(void);
void nick_join(int paikka);
void nick_kicked(void);
void nick_action(int offset);
void big_numbers(void);
void header(void);
void footer(void);
void htmlhead(void);
void nick_exclude(void);
void loadHistory(void);
void saveHistory(void);
void printTopic(void);
int searchNick(char sNick[NICKMAX], int connect, int from, int casesens);
void removeHtml(char html_txt[], int html_len, int isTopic);
void nickjoin(void);
void sort_nick(void);
void zero_nick(int znick);
void dayactrotate(void);
void gapfiller(char inputtxt[MAXLINE], char keyword[64]);
void getconf(char keyword[64], int fillgaps);
void timeofday(void);
int timetodate(int daycorrection, int isShort);


/* Globaalit muuttujat */
char cLine[MAXLINE];		/* current line */
char temp_nick[NICKMAX];
char langtemp[MAXLINE];
char confline[MAXLINE];
char CONFIG[256];
char USERDB[128];
char topic[5][256];
char fill[10][128];
char datestamp[10];

int debug, daystart;
int topickayt, totalNicks;
long countedLines, totalLines;
int dotcounter, day, dot;
int prevTime;
int h[24], runtime, tzone, usersInUse;
int dayact[31][4];
int continueFromLine;
char oldlastline[MAXLINE];
int sortCounter=0;

FILE *logfile, *html, *configfile;
char configInUse[256];

/* this limits the maximum number of different nicks */
USER user[NICKLIMIT];

time_t starttime, endtime;

/* Config variables */
char IHTML[256];
char EXNICK[256];
char JNICK[256];
char BGPIC[256];
char HEADER[256];
char FOOTER[256];
char COLORS[256];
char LANG[256];
char HHEAD[256];
char creatornick[20];
char channelname[40];
int nickconnect;
int casesensitive;
int njcasesensitive;
int topnbr;
int quotenbr;
int quotemin;
int timezoneCorrection;
int daycorrection;
int makehistory;
int lowtalk;
int scaleLinePipes;
char tablewidth[10];
char cbg[8];
char ct1[8];
char ct2[8];
char ct3[8];
char ct4[8];
char ct5[8];
char cf1[8];
char cf2[8];
char cf3[8];
char cf4[8];
char globalselect[16];

int main(int argc, char *argv[])
{
	long i;
	int j;
	int contLine, offset=0, offsetfound=0;

	if ((argc == 1) || (argc >= 4)) {
		printf("IRCStats %s by Teemu Toivola <tst at iki dot fi>\n\n", IVERSION);
		printf("Usage: ./ircstats <logfile> [configfile]\n");
		printf("Example: ./ircstats channel.log channel.cfg\n");
		exit(0);
	}

	if (argc == 3)
		strcpy(CONFIG,argv[2]);
	else
		strcpy(CONFIG,"ircstats.cfg");

	starttime=time(NULL);

	for (i=0;i<=23;i++) {
		h[i]=0;
	}

	for (i=0;i<=30;i++) {
		for (j=0;j<=3;j++)
			dayact[i][j]=0;
	}

	i=j=0;
	day=1;
	totalLines=0;
	usersInUse=-1;
	topickayt=-1;
	dot=0;
	continueFromLine=0;
	debug=0;

	strcpy(temp_nick,".....");

	if ((logfile=fopen(argv[1],"r"))==NULL) {
		fprintf(stderr,"\nError:\nUnable to open file \"%s\" for reading the log!\n\n", argv[1]);
		exit(1);
	}

	loadConfig();

	prevTime=0;

	countLines();

	if (countedLines==0) {
		printf("Logfile too short for analysis.\n");
		exit(1);
	}

	rewind(logfile);

	srand(starttime-countedLines);

	if (makehistory==1)
		loadHistory();

	contLine=0;

	if ((countedLines>=continueFromLine) && (makehistory==1)) {
		if (seekline(continueFromLine-1)==1) {
			contLine=continueFromLine-1;
			printf("Starting from line %d\n",contLine);
		} else {
			rewind(logfile);
			printf("Starting from first line.\n");
		}
	}

	if (makehistory==1) {
		totalLines=totalLines+countedLines-contLine;
	} else {
		totalLines=totalLines+countedLines;
	}

	dotcounter=countedLines/80;

	if (debug)
		printf("Analyzing log...\n\n");
	if (debug==1)
		printf("|0%%--------------25%%----------------50%%------------------75%%---------------100%%|\n");

	for (i=contLine;i<countedLines;i++) {
		if (fgets(cLine,MAXLINE,logfile)==NULL)
			break;
		if (debug==2)
			printf("%s",cLine);
		if (cLine[0]=='[') {
			if (!offsetfound) {
				if (cLine[0]=='[' && cLine[6]==']') {
					offset=0;
					offsetfound=1;
				} else if (cLine[0]=='[' && cLine[9]==']') {
					offset=3;
					offsetfound=1;
				} else {
				
					printDots(i);
					continue;
				}
			}
			parseTime();
			if (cLine[8+offset]==42 && cLine[9+offset]==42)
				parseAction(offset);
			if (cLine[8+offset]=='*' && cLine[9+offset]==' ')
				nick_action(offset);
			if (cLine[8+offset]=='<')
				parseChatline(offset);
			sortCounter++;
			if (sortCounter>=SORTFREQ) {
				sort_nick();
				sortCounter=0;
			}
		}
		printDots(i);
	}

	fclose(logfile);
	if (debug)
		printf("\n");

	if ((html=fopen(IHTML,"w"))==NULL) {
		fprintf(stderr,"\nError:\nUnable to write the requested html to file \"%s\"!\n\n",IHTML);
		exit(1);
	}

	if (debug==2)
		printf("Sorting nicks\n");
	sort_nick();
	if (makehistory==1) {
		if (debug==2)
			printf("Saving history data\n");
		saveHistory();
	}
	if (globalselect[2]!='0') {
		if (debug==2)
			printf("Nickjoin\n");
		nickjoin();
	}
	if (globalselect[1]!='0') {
		if (debug==2)
			printf("Exclude\n");
		nick_exclude();
	}
	if (debug==2)
		printf("Sorting nicks\n");
	sort_nick();
	if ((configfile=fopen(LANG,"r"))==NULL) {
		fprintf(stderr,"\nError:\nUnable to open file \"%s\" for reading configs!\n\n",LANG);
		exit(1);
	}
	strcpy(configInUse,LANG);
	if (debug==2)
		printf("Html header\n");
	startHtml();
	if (debug==1) {
		printf("Generating html... ");
		fflush(stdout);
	}

	if (lowtalk!=0) {
		for (j=0;j<=usersInUse;j++) {
			if (user[j].ttalk<lowtalk)
				zero_nick(j);
		}
	}
	if (globalselect[0]!='0') {
		if (debug==2)
			printf("Daily IRCing... and When do we...\n");
		timetable();
	}

	if (debug==2)
		printf("Most active nicks\n");
	printNicks();
	if (globalselect[8]!='0') {
		if (totalNicks>10) {
			if (debug==2)
				printf("Time of day\n");
			timeofday();
		}
	}
	if (globalselect[3]!='0') {
		if (topickayt>=0) {
			if (debug==2)
				printf("Topics\n");
			printTopic();
		}
	}
	if (globalselect[4]!='0') {
		if (usersInUse>0) {
			if (debug==2)
				printf("Big numbers\n");
			big_numbers();
		}
	}
	if (debug==2)
		printf("Final lines\n");
	footer();
	if (globalselect[5]!='0')
		longEnd();
	else
		shortEnd();
	fclose(configfile);
	fclose(html);
	if (debug==1)
		printf("done.\n\n");
	if (debug==2)
		printf("All done\n\n");
	return 0;
}


/* -------- */
void countLines(void)
{
	countedLines=0;
	totalLines=0;
	if (debug) {
		printf("\nCounting lines in logfile... ");
		fflush(stdout);
	}
	while (fgets(cLine,MAXLINE,logfile)!=NULL) {
		countedLines++;
	}
	if (debug)
		printf("%ld lines found.\n\n", countedLines);
}

int seekline(long line)
{
	long i;

	for (i=0;i<=line;i++)
		if (fgets(cLine,MAXLINE,logfile)==NULL)
			break;

	if (strncmp(oldlastline,cLine,512)==0) {
		if (countedLines==continueFromLine) {
			printf("Warning:\nNo new lines found from the logfile. Exiting.\n\n");
			exit(0);
		}
		return 1;
	}
	return 0;
}

void printDots(long linecounter)
{
	if (linecounter>dotcounter) {
		if (dot!=80) {
			if (debug==1)
				printf(".");
			dot++;
		}
		if (debug==1)
			fflush(stdout);
		dotcounter=dotcounter+countedLines/80;
	}
}

void parseTime(void)
{
	char currtime[3];
	int selection;
	selection=0;
	tzone=0;

	currtime[0]='0';
	currtime[1]='0';
	currtime[2]='\0';

	currtime[0]=cLine[1];
	currtime[1]=cLine[2];

	selection=atoi(currtime);

	selection=selection+timezoneCorrection;
	if (selection>=24)
		selection=selection-24;
	if (selection<=-1)
		selection=selection+24;

	if (selection<prevTime) {
		day++;
		dayactrotate();
	}

	if (debug==2) {
		if (selection<prevTime)
			printf("Time: %d -> %d (New day)\n",prevTime,selection);
		else
			printf("Time: %d -> %d\n",prevTime,selection);
	}

	switch (selection) {
		case 0: h[0]++;
			tzone=0;
			 break;
		case 1: h[1]++;
			tzone=0;
			 break;
		case 2: h[2]++;
			tzone=0;
			 break;
		case 3: h[3]++;
			tzone=0;
			 break;
		case 4: h[4]++;
			tzone=0;
			 break;
		case 5: h[5]++;
			tzone=0;
			 break;
		case 6: h[6]++;
			tzone=1;
			 break;
		case 7: h[7]++;
			tzone=1;
			 break;
		case 8: h[8]++;
			tzone=1;
			 break;
		case 9: h[9]++;
			tzone=1;
			 break;
		case 10: h[10]++;
			tzone=1;
			 break;
		case 11: h[11]++;
			tzone=1;
			 break;
		case 12: h[12]++;
			tzone=2;
			 break;
		case 13: h[13]++;
			tzone=2;
			 break;
		case 14: h[14]++;
			tzone=2;
			 break;
		case 15: h[15]++;
			tzone=2;
			 break;
		case 16: h[16]++;
			tzone=2;
			 break;
		case 17: h[17]++;
			tzone=2;
			 break;
		case 18: h[18]++;
			tzone=3;
			 break;
		case 19: h[19]++;
			tzone=3;
			 break;
		case 20: h[20]++;
			tzone=3;
			 break;
		case 21: h[21]++;
			tzone=3;
			 break;
		case 22: h[22]++;
			tzone=3;
			 break;
		case 23: h[23]++;
			tzone=3;
			 break;
		default: break;
	}
	prevTime=selection;

	dayact[0][tzone]++;

}

void startHtml(void)
{
	int j;

	fprintf(html,"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">\n\n");
	fprintf(html,"<!-- This page was created with IRCStats %s -->\n", IVERSION);
	fprintf(html,"<!-- IRCStats is &copy %s by Teemu Toivola -->\n", IYEARS);
	fprintf(html,"<!-- Do not modify the page source!  (Logged lines: %ld) -->\n\n",totalLines);

	fprintf(html,"<!-- IRCStats WWW: http://humdi.net/ircstats/ -->\n\n");

	fprintf(html,"<!-- Page design from mIRCStats by Mikko Auvinen. -->\n");
	fprintf(html,"<!-- mIRCStats WWW: http://www.nic.fi/~mauvinen/mircstats/ -->\n\n");

	strcpy(fill[0],channelname);
	strcpy(fill[1],creatornick);
	getconf("title1:",1);
	fprintf(html,"<html><head>\n<title>%s</title>\n",langtemp);
	htmlhead();
	fprintf(html,"</head>\n");
	if ((strlen(BGPIC))==1)
		fprintf(html,"<body bgcolor=\"%s\" text=\"%s\" link=\"%s\" vlink=\"%s\" alink=\"%s\"><center>\n",cbg,cf1,cf1,cf1,cf1);
	else
		fprintf(html,"<body background=\"%s\" bgcolor=\"%s\" text=\"%s\" link=\"%s\" vlink=\"%s\" alink=\"%s\"><center>\n",BGPIC,cbg,cf1,cf1,cf1,cf1);
	getconf("title2:",1);
	fprintf(html,"<font color=\"%s\" size=\"5\" face=\"Verdana\"><i>%s</i></font><br>\n",cf1,langtemp);
	fprintf(html,"<hr width=\"50%%\">");

	/* count how many nicks have lines */
	totalNicks=0;
	for (j=0;j<=usersInUse;j++) {
		if (user[j].ttalk!=0)
			totalNicks++;
		else
			break;
	}

	endtime=time(NULL);
	starttime=starttime+(3600*timezoneCorrection);
	endtime=endtime+(3600*timezoneCorrection);
	runtime=endtime-starttime;

	if (globalselect[9]!='0') {
		timetodate(day+daycorrection-1,0);
		strcpy(fill[1],datestamp);
		timetodate(daycorrection,0);
		strcpy(fill[2],datestamp);
		getconf("head3:",1);

	} else {
		strcpy(fill[1],(char*)asctime(localtime(&endtime)));
		getconf("head1:",1);
	}
	fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\">%s</font><br>\n",cf2,langtemp);
	sprintf(fill[1],"%d",day);
	sprintf(fill[2],"%d",totalNicks);
	getconf("head2:",1);
	fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\">%s</font><br>\n\n",cf2,langtemp);

	header();

	if (debug) {
		printf("\nSo far %d days has been logged.\n",day);
		if (runtime<=120)
			printf("The analysis took %d seconds.\n\n",runtime);
		else
			printf("The analysis took %d minutes.\n\n",runtime/60);
	}
}

void timetable(void)
{
	float pros, temp, isoin;
	int j, palkki, isoinj, disoin, dayactyht;

	palkki=100; /* height of bars */
	isoinj=0;
	disoin=0;
	pros=0.0;
	pros=totalLines/100.0;

	if (globalselect[7]!='0') { /* globalselect begins */
	for (j=0;j<=30;j++) {
		dayactyht=dayact[j][0]+dayact[j][1]+dayact[j][2]+dayact[j][3];
		if(dayactyht>disoin) {
			disoin=dayactyht;
			isoinj=j;
		}
	}

	fprintf(html,"<hr width=\"95%%\">\n");
	fprintf(html,"<font color=\"%s\" size=\"4\" face=\"Verdana\">",cf1);
	getconf("day1:",1);
	fprintf(html,"<i>%s</i></font><br>\n",langtemp);
	if (globalselect[9]=='0') {
		fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\">",cf2);
		getconf("day2:",1);
		fprintf(html,"<i>%s</i></font><br>\n",langtemp);
	}
	fprintf(html,"<table border=\"0\"><tr>\n");

	for (j=30;j>=0;j--) {
		dayactyht=dayact[j][0]+dayact[j][1]+dayact[j][2]+dayact[j][3];
		if(dayactyht) {
			fprintf(html,"<td align=\"center\" valign=\"bottom\" width=\"19\">");
			fprintf(html,"<font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%d<br>",cf2,dayactyht);

			temp=dayact[j][3];
			if((temp/disoin*palkki)>=0.5)
				fprintf(html,"<img src=\"pipe4v.gif\" alt=\"\" width=\"19\" height=\"%.0f\"><br>",temp/disoin*palkki);

			temp=dayact[j][2];
			if((temp/disoin*palkki)>=0.5)
				fprintf(html,"<img src=\"pipe3v.gif\" alt=\"\" width=\"19\" height=\"%.0f\"><br>",temp/disoin*palkki);

			temp=dayact[j][1];
			if((temp/disoin*palkki)>=0.5)
				fprintf(html,"<img src=\"pipe2v.gif\" alt=\"\" width=\"19\" height=\"%.0f\"><br>",temp/disoin*palkki);

			temp=dayact[j][0];
			if((temp/disoin*palkki)>=0.5)
				fprintf(html,"<img src=\"pipe1v.gif\" alt=\"\" width=\"19\" height=\"%.0f\"><br>",temp/disoin*palkki);

			fprintf(html,"</font></td>\n");
		}
	}
	fprintf(html,"</tr><tr>\n");
	for (j=30;j>=0;j--) {
		dayactyht=dayact[j][0]+dayact[j][1]+dayact[j][2]+dayact[j][3];
		if(dayactyht) {
			if (globalselect[9]!='0') {
				if(timetodate(j+daycorrection,1))
					fprintf(html,"<td bgcolor=\"%s\" align=\"center\" valign=\"bottom\">",ct2);
				else
					fprintf(html,"<td bgcolor=\"%s\" align=\"center\" valign=\"bottom\">",ct1);

				fprintf(html,"<font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%s</font></td>\n",cf3,datestamp);
			} else {
				if(j!=isoinj) /* (j!=isoinj) */  
					fprintf(html,"<td bgcolor=\"%s\" align=\"center\" valign=\"bottom\">",ct1);
				else
					fprintf(html,"<td bgcolor=\"%s\" align=\"center\" valign=\"bottom\">",ct2);

				fprintf(html,"<font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%d</font></td>\n",cf3,j);
			}
		}
	}
	fprintf(html,"</tr></table><br>\n");
	} /* globalselect ends */

	palkki=130;
	isoin=0.0;
	isoinj=0;
	pros=0.0;
	pros=totalLines/100.0;

	for (j=0;j<=23;j++) {
		if (h[j]/pros>isoin) {
			isoin=h[j]/pros;
			isoinj=j;
		}
	}

	fprintf(html,"<hr width=\"95%%\">\n");

	fprintf(html,"<table border=\"0\"><tr>\n");
	if (daystart==0) {
		getconf("pink:",1);
		fprintf(html,"<td width=\"180\"><img src=\"pipe1.gif\" height=\"15\" width=\"30\" align=top alt=\"%s\">",langtemp);
		getconf("hours0-6:",1);
		fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\"> %s</font></td>\n",cf1,langtemp);
	}
	getconf("green:",1);
	fprintf(html,"<td width=\"180\"><img src=\"pipe2.gif\" height=\"15\" width=\"30\" align=top alt=\"%s\">",langtemp);
	getconf("hours6-12:",1);
	fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\"> %s</font></td>\n",cf1,langtemp);
	getconf("blue:",1);
	fprintf(html,"<td width=\"180\"><img src=\"pipe3.gif\" height=\"15\" width=\"30\" align=top alt=\"%s\">",langtemp);
	getconf("hours12-18:",1);
	fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\"> %s</font></td>\n",cf1,langtemp);
	getconf("red:",1);
	fprintf(html,"<td width=\"180\"><img src=\"pipe4.gif\" height=\"15\" width=\"30\" align=top alt=\"%s\">",langtemp);
	getconf("hours18-24:",1);
	fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\"> %s</font></td>\n",cf1,langtemp);
	if (daystart==1) {
		getconf("pink:",1);
		fprintf(html,"<td width=\"180\"><img src=\"pipe1.gif\" height=\"15\" width=\"30\" align=top alt=\"%s\">",langtemp);
		getconf("hours0-6:",1);
		fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\"> %s</font></td>\n",cf1,langtemp);
	}

	fprintf(html,"</tr></table>\n");
	fprintf(html,"<hr width=\"95%%\">\n");

	getconf("when:",1);
	fprintf(html,"<font color=\"%s\" size=\"4\" face=\"Verdana\"><i>%s</i></font><br>\n",cf1,langtemp);
	fprintf(html,"<table border=\"0\"><tr>\n");

	if (daystart==0) {
		for (j=0;j<=5;j++) {
			temp=h[j]/pros;
			fprintf(html,"<td align=\"center\" valign=\"bottom\"><font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%.1f%%<br><img src=\"pipe1v.gif\" alt=\"\" width=\"18\" height=\"%.0f\"></font></td>\n",cf2,temp,temp/isoin*palkki);
		}
	}

	for (j=6;j<=11;j++) {
		temp=h[j]/pros;
		fprintf(html,"<td align=\"center\" valign=\"bottom\"><font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%.1f%%<br><img src=\"pipe2v.gif\" alt=\"\" width=\"18\" height=\"%.0f\"></font></td>\n",cf2,temp,temp/isoin*palkki);
	}

	for (j=12;j<=17;j++) {
		temp=h[j]/pros;
		fprintf(html,"<td align=\"center\" valign=\"bottom\"><font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%.1f%%<br><img src=\"pipe3v.gif\" alt=\"\" width=\"18\" height=\"%.0f\"></font></td>\n",cf2,temp,temp/isoin*palkki);
	}

	for (j=18;j<=23;j++) {
		temp=h[j]/pros;
		fprintf(html,"<td align=\"center\" valign=\"bottom\"><font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%.1f%%<br><img src=\"pipe4v.gif\" alt=\"\" width=\"18\" height=\"%.0f\"></font></td>\n",cf2,temp,temp/isoin*palkki);
	}

	if (daystart==1) {
		for (j=0;j<=5;j++) {
			temp=h[j]/pros;
			fprintf(html,"<td align=\"center\" valign=\"bottom\"><font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%.1f%%<br><img src=\"pipe1v.gif\" alt=\"\" width=\"18\" height=\"%.0f\"></font></td>\n",cf2,temp,temp/isoin*palkki);
		}
	}

	fprintf(html,"</tr><tr>\n");

	if (daystart==0) {
		for (j=0;j<=23;j++) {
			if (j!=isoinj)
				fprintf(html,"<td bgcolor=\"%s\" align=\"center\"><font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%-2d</font></td>\n",ct1,cf3,j);
			else
				fprintf(html,"<td bgcolor=\"%s\" align=\"center\"><font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%-2d</font></td>\n",ct2,cf3,j);
		}
	} else {
		for (j=6;j<=23;j++) {
			if (j!=isoinj)
				fprintf(html,"<td bgcolor=\"%s\" align=\"center\"><font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%-2d</font></td>\n",ct1,cf3,j);
			else
				fprintf(html,"<td bgcolor=\"%s\" align=\"center\"><font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%-2d</font></td>\n",ct2,cf3,j);
		}
		for (j=0;j<=5;j++) {
			if (j!=isoinj)
				fprintf(html,"<td bgcolor=\"%s\" align=\"center\"><font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%-2d</font></td>\n",ct1,cf3,j);
			else
				fprintf(html,"<td bgcolor=\"%s\" align=\"center\"><font color=\"%s\" size=\"1\" face=\"Arial Narrow\">%-2d</font></td>\n",ct2,cf3,j);
		}
	}
	fprintf(html,"</tr></table><br>\n");

}

void parseAction(int offset)
{
	int j, length, uustopic, ran;
	char foo[9][512];
	char temptopic[1024];

	memset(foo, '\0', sizeof(char)*9*512);
	length=strlen(cLine);

	for (j=0;j<length;j++) {
		if ((cLine[j]=='h') && (cLine[j+1]=='a') && (cLine[j+2]=='s') && (cLine[j+3]==' ') && (cLine[j+4]=='j')) {
			nick_join(1);
		}
		if ((cLine[j]=='J') && (cLine[j+1]=='o') && (cLine[j+3]=='n') && (cLine[j+4]=='s') && (cLine[j+5]==':')) {
			nick_join(2);
		}
		if ((cLine[j]=='w') && (cLine[j+1]=='a') && (cLine[j+2]=='s') && (cLine[j+3]==' ') && (cLine[j+4]=='k')) {
			nick_kicked();
		}
	}


	sscanf(cLine+11+offset,"%s %s %s %s %s %s %s %s %s",foo[0],foo[1],foo[2],foo[3],foo[4],foo[5],foo[6],foo[7],foo[8]);

	uustopic=0;

	/* IrcII style */
	if ((strcmp("changed",foo[2])==0) && (strcmp("topic",foo[4])==0)) {
		length=strlen(foo[0])+strlen(foo[7]);
		strcpy(temptopic,foo[0]);
		strcat(temptopic," ");
		strcat(temptopic,cLine+50+length);
		temptopic[strlen(temptopic)-1]='\0';
		if ((strcmp(topic[topickayt],temptopic)!=0) && (strcmp(foo[0],"*")!=0) && (strcmp(foo[0],"ChanServ")!=0))
			uustopic=1;
		/* printf("%s: %s",foo[0],cLine+50+length); */
	}

	/* mIRC style */
	if ((strcmp("changes",foo[1])==0) && (strcmp("topic",foo[2])==0)) {
		length=strlen(foo[0]);
		strcpy(temptopic,foo[0]);
		strcat(temptopic," ");
		strcat(temptopic,cLine+30+length);
		temptopic[strlen(temptopic)-1]='\0';
		if ((strcmp(topic[topickayt],temptopic)!=0) && (strcmp(foo[0],"*")!=0) && (strcmp(foo[0],"ChanServ")!=0))
			uustopic=1;
		/* printf("%s: %s",foo[0],cLine+30+length); */
	}

	if (globalselect[10]!='0') {
		if (uustopic==1) {
			topickayt++;
			if (topickayt==5) {
				ran=1+(int) (100.0*rand()/(RAND_MAX+1.0));
				if (ran>=65) {
					ran=((int) (100.0*rand()/(RAND_MAX+1.0))/20);
					strncpy(topic[ran],temptopic,255);
				}
				topickayt--;
			} else
				strncpy(topic[topickayt],temptopic,255);
			if (debug==2)
				printf("Topic: %s\n",temptopic);
		}
	} else {
		if (uustopic==1) {
			topickayt++;
			if (topickayt==5) {
				strcpy(topic[0],topic[1]);
				strcpy(topic[1],topic[2]);
				strcpy(topic[2],topic[3]);
				strcpy(topic[3],topic[4]);
				topickayt--;
			}
			strncpy(topic[topickayt],temptopic,255);
			if (debug==2)
				printf("Topic: %s\n",temptopic);
		}
	}

}

void parseChatline(int offset)
{
	int j, length, check, ran, linelen;
	int usernbr;
	char nikki[NICKMAX];

	sscanf(cLine+9+offset,"%s",nikki);
	length=strlen(nikki);
	linelen=strlen(cLine);
	nikki[length-1] = '\0';

	check=0;

	usernbr=searchNick(nikki, nickconnect, 0, casesensitive);

	if (usernbr!=-1)
		check=1;

	if (check == 0) {
		strcpy(user[usersInUse+1].nick,nikki);
		for (j=0;j<=3;j++)
			user[usersInUse+1].talk[j]=0;
		user[usersInUse+1].talk[tzone]=1;
		user[usersInUse+1].join=1;
		user[usersInUse+1].yell=0;
		user[usersInUse+1].ques=0;
		user[usersInUse+1].mono1=1;
		user[usersInUse+1].mono2=0;
		user[usersInUse+1].linelength=(float)linelen-11-offset-length;
		user[usersInUse+1].ttalk=1;
		user[usersInUse+1].action=0;
		user[usersInUse+1].kick=0;
		user[usersInUse+1].kicked=0;
		user[usersInUse+1].opped=0;
		user[usersInUse+1].deopped=0;
		user[usersInUse+1].happy=0;
		user[usersInUse+1].sad=0;
		strncpy(user[usersInUse+1].quote,cLine+10+offset+length,quotenbr);
		user[usersInUse+1].quote[strlen(user[usersInUse+1].quote)-1]='\0';
		usersInUse++;
		usernbr=usersInUse;
		if (debug==2)
			printf("New nick: %s\n",nikki);
		if (usersInUse==(NICKLIMIT-201))
			fprintf(stderr,"\nWarning:\nUserlimit will be reached after 200 new nicks!\n\n");
		if (usersInUse==(NICKLIMIT-1)) {
			fprintf(stderr,"\nError:\nUserlimit reached, unable to continue.\n");
			fprintf(stderr,"See the README for more info.\n\n");
			exit(1);
		}

	}

	if (check == 1) {
		if ((strcmp(temp_nick,nikki)) == 0)
			user[usernbr].mono1++;
		else
			user[usernbr].mono1=1;
		user[usernbr].talk[tzone]++;
		user[usernbr].ttalk++;
		user[usernbr].linelength=((user[usernbr].linelength*(user[usernbr].ttalk-1))+(linelen-11-offset-length))/user[usernbr].ttalk;

		ran=1+(int) (100.0*rand()/(RAND_MAX+1.0));
		if (debug==2)
			printf("Quote rnd: %d",ran);
		if ((ran>=90) && (strlen(cLine+10+offset+length)>=quotemin) && (strlen(cLine+10+offset-length)<=quotenbr)) {
			strncpy(user[usernbr].quote,cLine+10+offset+length,quotenbr);
			user[usernbr].quote[strlen(user[usernbr].quote)-1]='\0';
			if (debug==2)
				printf(" - OK");
		}
		if (debug==2)
			printf("\n");

		if (user[usernbr].mono1 == 5) {
			user[usernbr].mono2++;
			user[usernbr].mono1=1;
			if (debug==2)
				printf("Monologue: %s\n",user[usernbr].nick);
		}
	}
	strcpy(temp_nick,nikki);

	if ((cLine[linelen-3]==33) || (cLine[linelen-2]==33))
		user[usernbr].yell++;
	if ((cLine[linelen-3]==63) || (cLine[linelen-2]==63))
		user[usernbr].ques++;

	if ( ((cLine[linelen-3]==':') || (cLine[linelen-3]=='=')) && ((cLine[linelen-2]==')') || (cLine[linelen-2]=='D')) )
		user[usernbr].happy++;
	else if ( ((cLine[linelen-3]==':') || (cLine[linelen-3]=='=')) && ((cLine[linelen-2]=='(') || (cLine[linelen-2]=='/')) )
		user[usernbr].sad++;

	if ( ((cLine[linelen-2]==':') || (cLine[linelen-2]=='=')) && ((cLine[linelen-1]==')') || (cLine[linelen-1]=='D')) )
		user[usernbr].happy++;
	else if ( ((cLine[linelen-2]==':') || (cLine[linelen-2]=='=')) && ((cLine[linelen-1]=='(') || (cLine[linelen-1]=='/')) )
		user[usernbr].sad++;
}

void printNicks(void)
{
	int j, a, tablecount;
	float palkkiv, palkki, pitrivi, pitpalkki;
	float pv[4];

	fprintf(html,"<hr width=\"95%%\">\n");
	getconf("top1:",1);
	fprintf(html,"<font color=\"%s\" size=\"4\" face=\"Verdana\"><i>%s</i></font><br>",cf1,langtemp);
	getconf("top2:",1);
	fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\"><i>%s</i></font>\n",cf2,langtemp);
	if ((atoi(tablewidth))!=0)
		fprintf(html,"<table border=\"0\" cellpadding=\"2\" cellspacing=\"2\" width=\"%s\">\n",tablewidth);
	else
		fprintf(html,"<table border=\"0\" cellpadding=\"2\" cellspacing=\"2\">\n");
	getconf("nick:",1);
	fprintf(html,"<tr><td>&nbsp;</td><td align=\"left\" bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\"><i><b>%s</b></i></font></td>\n",ct3,cf3,langtemp);
	getconf("line:",1);
	fprintf(html,"<td align=\"left\" bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\"><i><b>%s</b></i></font></td>\n",ct3,cf3,langtemp);
	if (globalselect[6]!='0') {
		getconf("average:",1);
		fprintf(html,"<td align=\"left\" bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\"><i><b>%s</b></i></font></td>\n",ct3,cf3,langtemp);
	}
	getconf("quote:",1);
	fprintf(html,"<td align=\"left\" bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\"><i><b>%s</b></i></font></td></tr>\n",ct3,cf3,langtemp);

	pitrivi=0.0;
	for (j=0;j<=topnbr;j++) {
		if (user[j].linelength>pitrivi) {
			pitrivi=user[j].linelength;
		}
	}

	palkki=100.0;
	for (a=0;a<=topnbr;a++) {
		if (user[a].ttalk==0)
			break;
		if(scaleLinePipes)
			palkkiv=(user[a].ttalk/(float)user[0].ttalk)*palkki;
		else
			palkkiv=70.0;
		pitpalkki=(user[a].linelength/pitrivi)*palkki;
		for (j=0;j<=3;j++)
			pv[j]=(user[a].talk[j]/(float)user[a].ttalk)*palkkiv;
		fprintf(html,"<tr><td align=\"right\" bgcolor=\"%s\"><font color=\"%s\" size=\"1\" face=\"Verdana\">%d</font></td>\n",ct4,cf4,a+1);
		if (a==0)
			fprintf(html,"<td align=\"left\" bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\"><b>%s</b></font></td>\n",ct5,cf4,user[a].nick);
		else
			fprintf(html,"<td align=\"left\" bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\">%s</font></td>\n",ct5,cf4,user[a].nick);

		fprintf(html,"<td align=\"left\" valign=\"middle\" bgcolor=\"%s\">",ct5);
		if ((int)pv[0]!=0)
			fprintf(html,"<img src=\"pipe1.gif\" alt=\"\" height=\"15\" width=\"%.0f\" align=top>",pv[0]);
		if ((int)pv[1]!=0)
			fprintf(html,"<img src=\"pipe2.gif\" alt=\"\" height=\"15\" width=\"%.0f\" align=top>",pv[1]);
		if ((int)pv[2]!=0)
			fprintf(html,"<img src=\"pipe3.gif\" alt=\"\" height=\"15\" width=\"%.0f\" align=top>",pv[2]);
		if ((int)pv[3]!=0)
			fprintf(html,"<img src=\"pipe4.gif\" alt=\"\" height=\"15\" width=\"%.0f\" align=top>",pv[3]);
		if (a==0)
			fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\"><b> %d</b></font></td>\n",cf4,user[a].ttalk);
		else
			fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\"> %d</font></td>\n",cf4,user[a].ttalk);
		if (globalselect[6]!='0') {
			if (user[a].linelength==pitrivi)
				fprintf(html,"<td align=\"left\" bgcolor=\"%s\"><img src=\"pipe5.gif\" alt=\"\" height=\"15\" width=\"%.0f\" align=top><font color=\"%s\" size=\"2\" face=\"Verdana\"><b> %.0f</b></font></td>\n",ct5,pitpalkki,cf4,user[a].linelength);
			else
				fprintf(html,"<td align=\"left\" bgcolor=\"%s\"><img src=\"pipe5.gif\" alt=\"\" height=\"15\" width=\"%.0f\" align=top><font color=\"%s\" size=\"2\" face=\"Verdana\"> %.0f</font></td>\n",ct5,pitpalkki,cf4,user[a].linelength);
		}
		removeHtml(user[a].quote, strlen(user[a].quote), 0);
		fprintf(html,"<td align=\"left\" bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\"><i> \"%s\"</i></font></td>\n",ct5,cf4,user[a].quote);
		fprintf(html,"</tr>\n");
	}

	fprintf(html,"</table>\n\n");

	/* small top */

	if ((a>topnbr) && (user[a].ttalk!=0)) {
		getconf("stop:",1);
		fprintf(html,"<br><font color=\"%s\" size=\"2\" face=\"Verdana\"><b>%s</b></font><br>\n",cf1,langtemp);
		fprintf(html,"<table border=\"0\" cellpadding=\"2\" cellspacing=\"2\">\n");
		fprintf(html,"<tr>\n");

		tablecount=0;
		for (;a<=topnbr+SMALLTOP;a++) {
			if (user[a].ttalk==0)
				break;
			fprintf(html,"<td align=\"left\" bgcolor=\"%s\"><font color=\"%s\" size=\"1\" face=\"Verdana\"> %s (%d) </font></td>\n",ct5,cf4,user[a].nick,user[a].ttalk);
			tablecount++;
			if ((tablecount==6) && (a+1<=topnbr+SMALLTOP)) {
				fprintf(html,"</tr>\n<tr>\n");
				tablecount=0;
			}
		}
		fprintf(html,"</tr>\n</table>");
		if (user[a].ttalk) {
			sprintf(fill[1],"%d",totalNicks-topnbr-SMALLTOP-1);
			getconf("stop2:",1);
			fprintf(html,"<br><font color=\"%s\" size=\"2\" face=\"Verdana\">%s</font><br>\n",cf2,langtemp);
		}
	}
	fprintf(html,"<br>\n");

}

void longEnd(void)
{
	int i;

	/* search where the path part ends */
	for (i=strlen(LANG);i!=0;i--) {
		if (LANG[i]=='/') {
			i++;
			break;
		}
	}

	/* if you edit this then leave at least a link to the IRCStats homepage */
	fprintf(html,"<br><table width=\"95%%\" border=\"2\"><tr><td bgcolor=\"#E0E0F0\"><table width=\"100%%\"  border=\"0\" cellpadding=\"5\" cellspacing=\"0\">\n");
	fprintf(html,"<tr><td align=\"left\" valign=\"middle\"><font color=\"#202000\" size=\"2\" face=\"Verdana\">This page was created on %s with IRCStats %s.</font><br>\n",asctime(localtime(&endtime)), IVERSION);
	if (runtime<=120)
		fprintf(html,"<font color=\"#002040\" size=\"1\" face=\"Verdana\">Run time: %d seconds. Language file used: \"%s\".</font></td>\n",runtime,LANG+i);
	else
		fprintf(html,"<font color=\"#002040\" size=\"1\" face=\"Verdana\">Run time: %d minutes. Language file used: \"%s\".</font></td>\n",runtime/60,LANG+i);
	fprintf(html,"<td align=\"right\" valign=\"middle\"><font color=\"#300000\" size=\"2\" face=\"Verdana\">IRCStats is &copy %s by Teemu Toivola</font><br>\n", IYEARS);
	fprintf(html,"<font color=\"#300000\" size=\"2\" face=\"Verdana\">Download the latest version from: </font><a href=\"http://humdi.net/ircstats/\" target=\"_top\"><font color=\"#100000\" size=\"2\" face=\"Verdana\"><b>IRCStats homepage</b></font></a></td>\n");
	fprintf(html,"</tr></table></td></tr></table>\n");
	fprintf(html,"</center></body></html>\n");
}

void shortEnd(void)
{
	/* if you edit this then leave at least a link to the IRCStats homepage */
	fprintf(html,"<br><table border=\"2\"><tr><td bgcolor=\"#E0E0F0\"><table width=\"100%%\"  border=\"0\" cellpadding=\"5\" cellspacing=\"0\">\n");
	fprintf(html,"<tr><td align=\"left\" valign=\"middle\"><font color=\"#202000\" size=\"2\" face=\"Verdana\">This page was created with <a href=\"http://humdi.net/ircstats/\" target=\"_top\"><font color=\"#202000\" size=\"2\" face=\"Verdana\">IRCStats</font></a> 1.0.<br>\n");
	fprintf(html,"</td></tr></table></td></tr></table>\n");
	fprintf(html,"</center></body></html>\n");
}

void loadConfig(void)
{
	if ((configfile=fopen(CONFIG,"r"))==NULL) {
		fprintf(stderr,"\nError:\nUnable to open file \"%s\" for reading configs!\n\n",CONFIG);
		exit(1);
	}
	strcpy(configInUse,CONFIG);

	/* Output level */
	getconf("outputLevel:",0);
	debug=atoi(confline);

	if (debug==2)
		printf("\nWarning:\nDebug is ON!\n\n");

	/* the old thing known as global select... */
	getconf("showWhenDoWe:",0);
	globalselect[0]=confline[0];
	getconf("useNickexclude:",0);
	globalselect[1]=confline[0];
	getconf("useNickjoin.txt:",0);
	globalselect[2]=confline[0];
	getconf("showTopic:",0);
	globalselect[3]=confline[0];
	getconf("showBigNumbers:",0);
	globalselect[4]=confline[0];
	getconf("showLongEndText:",0);
	globalselect[5]=confline[0];
	getconf("showAverageChar/Line:",0);
	globalselect[6]=confline[0];
	getconf("showDailyIRCingActivity:",0);
	globalselect[7]=confline[0];
	getconf("showTimeOfDay:",0);
	globalselect[8]=confline[0];
	getconf("showDatesInDailyIRCingActivity:",0);
	globalselect[9]=confline[0];
	getconf("useRandomTopics:",0);
	globalselect[10]=confline[0];

	/* filename for output */
	getconf("htmlFilename:",0);
	strcpy(IHTML,confline);

	/* language filename */
	getconf("languageFilename:",0);
	strcpy(LANG,confline);

	/* filename for the head-part of the html code */
	getconf("htmlheadFilename:",0);
	strcpy(HHEAD,confline);

	/* exclude filename */
	getconf("excludeFilename:",0);
	strcpy(EXNICK,confline);

	/* nickjoin filename */
	getconf("nickjoinFilename:",0);
	strcpy(JNICK,confline);

	/* header filename */
	getconf("headerFilename:",0);
	strcpy(HEADER,confline);

	/* footer filename */
	getconf("footerFilename:",0);
	strcpy(FOOTER,confline);

	/* filename for the background picture */
	getconf("backgroundFilename:",0);
	strcpy(BGPIC,confline);

	/* colorcfg filename */
	getconf("colorsFilename:",0);
	strcpy(COLORS,confline);

	/* creator of the page */
	getconf("creatorOfLog:",0);
	strcpy(creatornick,confline);

	/* name of the channel */
	getconf("channelName:",0);
	strcpy(channelname,confline);

	/* filename for history data */
	getconf("historydataDir:",0);
	strcpy(USERDB,confline);
	strcat(USERDB,channelname);
	strcat(USERDB,".dat");

	/* how many letters are used to connect nicks */
	getconf("nickconnection:",0);
	nickconnect=atoi(confline);

	/* these names should be self-explaining anyway... */
	getconf("isNickconnectorCaseSensitive:",0);
	casesensitive=atoi(confline);

	/* ... */
	getconf("isNickjoin.txtCaseSensitive:",0);
	njcasesensitive=atoi(confline);

	/* ... */
	getconf("topTalkerLength:",0);
	topnbr=atoi(confline);
	topnbr--;

	/* ... */
	getconf("linesRequired:",0);
	lowtalk=atoi(confline);

	/* ... */
	getconf("topTalkerTableWidth:",0);
	strcpy(tablewidth,confline);

	/* ... */
	getconf("quoteLength:",0);
	quotenbr=atoi(confline);

	/* ... */
	getconf("quoteMinimum:",0);
	quotemin=atoi(confline);

	/* ... */
	getconf("timezoneCorrection:",0);
	timezoneCorrection=atoi(confline);

	/* ... */
	getconf("dayCorrection:",0);
	daycorrection=atoi(confline);

	/* ... */
	getconf("standardWhenDoWe:",0);
	daystart=atoi(confline);

	/* ... */
	getconf("useHistorySystem:",0);
	makehistory=atoi(confline);

	/* ... */
	getconf("scaleLinePipes:",0);
	scaleLinePipes=atoi(confline);

	fclose(configfile);

	if ((configfile=fopen(COLORS,"r"))==NULL) {
		fprintf(stderr,"\nError:\nUnable to open file \"%s\" for reading configs!\n\n",COLORS);
		exit(1);
	}
	strcpy(configInUse,COLORS);

	/* Colors */
	getconf("bgColor:",0);
	strcpy(cbg,confline);

	getconf("1stTColor:",0);
	strcpy(ct1,confline);

	getconf("2ndTColor:",0);
	strcpy(ct2,confline);

	getconf("3rdTColor:",0);
	strcpy(ct3,confline);

	getconf("4thTColor:",0);
	strcpy(ct4,confline);

	getconf("5thTColor:",0);
	strcpy(ct5,confline);

	getconf("1stFColor:",0);
	strcpy(cf1,confline);

	getconf("2ndFColor:",0);
	strcpy(cf2,confline);

	getconf("3rdFColor:",0);
	strcpy(cf3,confline);

	getconf("4thFColor:",0);
	strcpy(cf4,confline);

	fclose(configfile);
}

void nick_join(int paikka)
{
	int usernbr, check;
	char nikki[NICKMAX];

	check=0;
	if (paikka==1)
		sscanf(cLine+12,"%s",nikki);
	if (paikka==2)
		sscanf(cLine+19,"%s",nikki);

	usernbr=searchNick(nikki, nickconnect, 0, casesensitive);

	if (usernbr!=-1)
		check=1;

	if (check == 1) {
		user[usernbr].join++;
		if (debug==2)
			printf("Join: %s\n",nikki);
	}
}

void nick_kicked(void)
{
	int usernbr, check, length;
	char nikki[NICKMAX];

	check=0;
	sscanf(cLine+12,"%s",nikki);
	length=strlen(nikki);

	usernbr=searchNick(nikki, nickconnect, 0, casesensitive);

	if (usernbr!=-1)
		check=1;

	if (check == 1) {
		user[usernbr].kicked++;
		if (debug==2)
			printf("Kicked: %s\n",nikki);
	}

	/* who kicked */
	check=0;
	sscanf(cLine+length+27,"%s",nikki);
	length=strlen(nikki);

	usernbr=searchNick(nikki, nickconnect, 0, casesensitive);

	if (usernbr!=-1)
		check=1;

	if (check == 1) {
		user[usernbr].kick++;
		if (debug==2)
			printf("Kicker: %s\n",nikki);
	}

}

void nick_action(int offset)
{
	int usernbr, check;
	char nikki[NICKMAX];

	check=0;
	sscanf(cLine+10+offset,"%s",nikki);

	usernbr=searchNick(nikki, nickconnect, 0, casesensitive);

	if (usernbr!=-1)
		check=1;

	if (check == 1) {
		user[usernbr].action++;
		if (debug==2)
			printf("Action: %s\n",nikki);
	}
}

void big_numbers(void)
{

	int j, joinernickjoins, pisrivlaskin, kike, kike2, kik, kik2, act, act2, happylines;
	int joinernick=0, pisinrivinick=0, yellnick=0, yellnick2=0, happynick=0, happynick2=0, sadnick=0, sadnick2=0;
	int quesnick=0, quesnick2=0, kikenick=0, kikenick2=0, kiknick=0, kiknick2=0;
	int actnick=0, actnick2=0;
	float keskiarvo, pisinrivi, happy, happy2, sad, sad2, yell, yell2, ques, ques2;
	unsigned long ncountedLines;

	int mono, mono2;
	int mononick=0, mononick2=0;

	joinernickjoins=0;
	for (j=0;j<=usersInUse;j++) {
		if (user[j].join>joinernickjoins) {
			joinernickjoins=user[j].join;
			joinernick=j;
		}
	}

	pisinrivi=0;
	keskiarvo=0;
	pisrivlaskin=0;
	for (j=0;j<=usersInUse;j++) {
		if (user[j].ttalk>60) {
			keskiarvo=keskiarvo+user[j].linelength;
			pisrivlaskin++;
		}
		if ( (user[j].linelength>pisinrivi) && ( (user[j].ttalk/(float)user[0].ttalk*100>=5) || (j<=topnbr) ) ) {
			pisinrivi=user[j].linelength;
			pisinrivinick=j;
		}
	}
	keskiarvo=keskiarvo/(float)pisrivlaskin;

	/* yell */
	yell=0;
	yell2=0;
	yellnick=0;
	for (j=0;j<=usersInUse;j++) {
		if ((user[j].yell/(float)user[j].ttalk*100>=yell) && ( (user[j].ttalk/(float)user[0].ttalk*100>=5) || (j<=topnbr) )) {
			yell=user[j].yell/(float)user[j].ttalk*100;
			yellnick=j;
		}
	}
	for (j=0;j<=usersInUse;j++) {
		if ((user[j].yell/(float)user[j].ttalk*100>=yell2) && ( (user[j].ttalk/(float)user[0].ttalk*100>=5) || (j<=topnbr) ) && (user[j].yell/(float)user[j].ttalk*100<=yell) && (yellnick!=j)) {
			yell2=user[j].yell/(float)user[j].ttalk*100;
			yellnick2=j;
		}
	}


	/* questions */
	ques=0;
	ques2=0;
	quesnick=0;
	for (j=0;j<=usersInUse;j++) {
		if ((user[j].ques/(float)user[j].ttalk*100>=ques) && ( (user[j].ttalk/(float)user[0].ttalk*100>=5) || (j<=topnbr) )) {
			ques=user[j].ques/(float)user[j].ttalk*100;
			quesnick=j;
		}
	}
	for (j=0;j<=usersInUse;j++) {
		if ((user[j].ques/(float)user[j].ttalk*100>=ques2) && ( (user[j].ttalk/(float)user[0].ttalk*100>=5) || (j<=topnbr) ) && (user[j].ques/(float)user[j].ttalk*100<=ques) && (quesnick!=j)) {
			ques2=user[j].ques/(float)user[j].ttalk*100;
			quesnick2=j;
		}
	}

	/* monologues */
	mono=0;
	mono2=0;
	for (j=0;j<=usersInUse;j++) {
		if (user[j].mono2>mono) {
			mono=user[j].mono2;
			mononick=j;
		}
	}
	for (j=0;j<=usersInUse;j++) {
		if ((user[j].mono2>mono2) && (user[j].mono2<=mono) && (mononick!=j)) {
			mono2=user[j].mono2;
			mononick2=j;
		}
	}

	/* most kicked */
	kike=0;
	kike2=0;
	for (j=0;j<=usersInUse;j++) {
		if (user[j].kicked>kike) {
			kike=user[j].kicked;
			kikenick=j;
		}
	}
	for (j=0;j<=usersInUse;j++) {
		if ((user[j].kicked>kike2) && (user[j].kicked<=kike) && (kikenick!=j)) {
			kike2=user[j].kicked;
			kikenick2=j;
		}
	}

	/* most kicks */
	kik=0;
	kik2=0;
	for (j=0;j<=usersInUse;j++) {
		if (user[j].kick>kik) {
			kik=user[j].kick;
			kiknick=j;
		}
	}
	for (j=0;j<=usersInUse;j++) {
		if ((user[j].kick>kik2) && (user[j].kick<=kik) && (kiknick!=j)) {
			kik2=user[j].kick;
			kiknick2=j;
		}
	}

	/* actions */
	act=0;
	act2=0;
	for (j=0;j<=usersInUse;j++) {
		if (user[j].action>act) {
			act=user[j].action;
			actnick=j;
		}
	}
	for (j=0;j<=usersInUse;j++) {
		if ((user[j].action>act2) && (user[j].action<=act) && (actnick!=j)) {
			act2=user[j].action;
			actnick2=j;
		}
	}

	/* happy */
	happy=0;
	happy2=0;
	happylines=0;
	ncountedLines=0;
	for (j=0;j<=usersInUse;j++) {
		happylines+=user[j].happy;
		ncountedLines+=user[j].ttalk;
		if ((user[j].happy/(float)user[j].ttalk*100>=happy) && ( (user[j].ttalk/(float)user[0].ttalk*100>=5) || (j<=topnbr) )) {
			happy=user[j].happy/(float)user[j].ttalk*100;
			happynick=j;
		}
	}
	for (j=0;j<=usersInUse;j++) {
		if ((user[j].happy/(float)user[j].ttalk*100>=happy2) && ( (user[j].ttalk/(float)user[0].ttalk*100>=5) || (j<=topnbr) ) && (user[j].happy/(float)user[j].ttalk*100<=happy) && (happynick!=j)) {
			happy2=user[j].happy/(float)user[j].ttalk*100;
			happynick2=j;
		}
	}

	/* sad */
	sad=0;
	sad2=0;
	for (j=0;j<=usersInUse;j++) {
		if ((user[j].sad/(float)user[j].ttalk*100>=sad) && ( (user[j].ttalk/(float)user[0].ttalk*100>=5) || (j<=topnbr) )) {
			sad=user[j].sad/(float)user[j].ttalk*100;
			sadnick=j;
		}
	}
	for (j=0;j<=usersInUse;j++) {
		if ((user[j].sad/(float)user[j].ttalk*100>=sad2) && ( (user[j].ttalk/(float)user[0].ttalk*100>=5) || (j<=topnbr) ) && (user[j].sad/(float)user[j].ttalk*100<=sad) && (sadnick!=j)) {
			sad2=user[j].sad/(float)user[j].ttalk*100;
			sadnick2=j;
		}
	}

	/* html */
	fprintf(html,"<br><hr width=\"95%%\">\n");
	getconf("bign:",1);
	fprintf(html,"<font color=\"%s\" size=\"4\" face=\"Verdana\"><i>%s</i></font><br>\n",cf1,langtemp);
	fprintf(html,"<table border=\"0\" cellpadding=\"2\" cellspacing=\"3\" width=\"70%%\">\n");

	if (joinernickjoins!=1) {
		fprintf(html,"<tr><td bgcolor=\"%s\" valign=middle align=left><dl><dt><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct1,cf3);
		sprintf(fill[1],"<b><i>%s</i></b>",user[joinernick].nick);
		sprintf(fill[2],"%d",joinernickjoins);
		getconf("join:",1);
    	fprintf(html,"%s</font></dl></td></tr>\n",langtemp);
	}

	if (act2>=1) {
		fprintf(html,"<tr><td bgcolor=\"%s\" valign=middle align=left><dl><dt><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct1,cf3);
		sprintf(fill[1],"<b><i>%s</i></b>",user[actnick].nick);
		sprintf(fill[2],"%d",act);
		getconf("desc1:",1);
		fprintf(html,"%s</font></dt>\n",langtemp);
		sprintf(fill[1],"<b>%s</b>",user[actnick2].nick);
		sprintf(fill[2],"%d",act2);
		getconf("desc2:",1);
		fprintf(html,"<dt><font color=\"%s\" size=\"1\" face=\"Verdana\">%s</font></dt>\n",cf4,langtemp);
		fprintf(html,"</dl></td></tr>\n");
	}

	if (kik2>=1) {
		fprintf(html,"<tr><td bgcolor=\"%s\" valign=middle align=left><dl><dt><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct1,cf3);
		sprintf(fill[1],"<b><i>%s</i></b>",user[kiknick].nick);
		sprintf(fill[2],"%d",kik);
		getconf("evil1:",1);
		fprintf(html,"%s</font></dt>\n",langtemp);
		sprintf(fill[1],"<b>%s</b>",user[kiknick2].nick);
		sprintf(fill[2],"%d",kik2);
		getconf("evil2:",1);
		fprintf(html,"<dt><font color=\"%s\" size=\"1\" face=\"Verdana\">%s</font></dt>\n",cf4,langtemp);
		fprintf(html,"</dl></td></tr>\n");
	}

	if (kike2>=1) {
		fprintf(html,"<tr><td bgcolor=\"%s\" valign=middle align=left><dl><dt><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct1,cf3);
		sprintf(fill[1],"<b><i>%s</i></b>",user[kikenick].nick);
		sprintf(fill[2],"%d",kike);
		getconf("kick1:",1);
		fprintf(html,"%s</font></dt>\n",langtemp);
		sprintf(fill[1],"<b>%s</b>",user[kikenick2].nick);
		sprintf(fill[2],"%d",kike2);
		getconf("kick2:",1);
		fprintf(html,"<dt><font color=\"%s\" size=\"1\" face=\"Verdana\">%s</font></dt>\n",cf4,langtemp);
		fprintf(html,"</dl></td></tr>\n");
	}

	if (yell2) {
		fprintf(html,"<tr><td bgcolor=\"%s\" valign=middle align=left><dl><dt><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct1,cf3);
		sprintf(fill[1],"<b><i>%s</i></b>",user[yellnick].nick);
		sprintf(fill[2],"%.1f",yell);
		getconf("yell1:",1);
		fprintf(html,"%s</font></dt>\n",langtemp);
		sprintf(fill[1],"<b>%s</b>",user[yellnick2].nick);
		sprintf(fill[2],"%.1f",yell2);
		getconf("yell2:",1);
		fprintf(html,"<dt><font color=\"%s\" size=\"1\" face=\"Verdana\">%s</font></dt>\n",cf4,langtemp);
		fprintf(html,"</dl></td></tr>\n");
	}

	if (ques2) {
		fprintf(html,"<tr><td bgcolor=\"%s\" valign=middle align=left><dl><dt><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct1,cf3);
		sprintf(fill[1],"<b><i>%s</i></b>",user[quesnick].nick);
		sprintf(fill[2],"%.1f",ques);
		getconf("quest1:",1);
		fprintf(html,"%s</font></dt>\n",langtemp);
		sprintf(fill[1],"<b>%s</b>",user[quesnick2].nick);
		sprintf(fill[2],"%.1f",ques2);
		getconf("quest2:",1);
		fprintf(html,"<dt><font color=\"%s\" size=\"1\" face=\"Verdana\">%s</font></dt>\n",cf4,langtemp);
		fprintf(html,"</dl></td></tr>\n");
	}

	if (mono2>=1) {
		fprintf(html,"<tr><td bgcolor=\"%s\" valign=middle align=left><dl><dt><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct1,cf3);
		sprintf(fill[1],"<b><i>%s</i></b>",user[mononick].nick);
		sprintf(fill[2],"%d",mono);
		getconf("mono1:",1);
		fprintf(html,"%s</font></dt>\n",langtemp);
		sprintf(fill[1],"<b>%s</b>",user[mononick2].nick);
		sprintf(fill[2],"%d",mono2);
		getconf("mono2:",1);
		fprintf(html,"<dt><font color=\"%s\" size=\"1\" face=\"Verdana\">%s</font></dt>\n",cf4,langtemp);
		fprintf(html,"</dl></td></tr>\n");
	}

	if (happy2) {
		fprintf(html,"<tr><td bgcolor=\"%s\" valign=middle align=left><dl><dt><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct1,cf3);
		sprintf(fill[1],"<b><i>%s</i></b>",user[happynick].nick);
		sprintf(fill[2],"%.1f",happy);
		getconf("happy1:",1);
		fprintf(html,"%s</font></dt>\n",langtemp);
		sprintf(fill[1],"<b>%s</b>",user[happynick2].nick);
		sprintf(fill[2],"%.1f",happy2);
		getconf("happy2:",1);
		fprintf(html,"<dt><font color=\"%s\" size=\"1\" face=\"Verdana\">%s</font></dt>\n",cf4,langtemp);
		fprintf(html,"</dl></td></tr>\n");
	}

	if (sad2) {
		fprintf(html,"<tr><td bgcolor=\"%s\" valign=middle align=left><dl><dt><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct1,cf3);
		sprintf(fill[1],"<b><i>%s</i></b>",user[sadnick].nick);
		sprintf(fill[2],"%.1f",sad);
		getconf("sad1:",1);
		fprintf(html,"%s</font></dt>\n",langtemp);
		sprintf(fill[1],"<b>%s</b>",user[sadnick2].nick);
		sprintf(fill[2],"%.1f",sad2);
		getconf("sad2:",1);
		fprintf(html,"<dt><font color=\"%s\" size=\"1\" face=\"Verdana\">%s</font></dt>\n",cf4,langtemp);
		fprintf(html,"</dl></td></tr>\n");
	}

	if ((pisinrivi!=0) && (pisinrivi>keskiarvo)) {
		fprintf(html,"<tr><td bgcolor=\"%s\" valign=middle align=left><dl><dt><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct1,cf3);
		sprintf(fill[1],"<b><i>%s</i></b>",user[pisinrivinick].nick);
		sprintf(fill[2],"%.0f",pisinrivi);
		getconf("line1:",1);
		fprintf(html,"%s</font></dt>\n",langtemp);
		sprintf(fill[1],"%.0f",keskiarvo);
		getconf("line2:",1);
		fprintf(html,"<dt><font color=\"%s\" size=\"1\" face=\"Verdana\">%s</font></dt>\n",cf4,langtemp);
		fprintf(html,"</dl></td></tr>\n");
	}

	fprintf(html,"</table><br>\n");

	sprintf(fill[1],"%ld",totalLines);
	getconf("lines:",1);
	fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\"><i>%s</i></font><br>\n",cf2,langtemp);
}

void header(void)
{
	int openCheck;
	FILE *head;

	openCheck=0;
	if ((head=fopen(HEADER,"r"))!=NULL)
		openCheck=1;

	if (openCheck) {
		fprintf(html,"\n<hr width=\"50%%\">\n");
		fprintf(html,"<!-- User defined header begins -->\n");
		while(fgets(cLine,MAXLINE,head)!=NULL)
			fprintf(html,"%s",cLine);
		fprintf(html,"<!-- User defined header ends -->\n\n");
		fclose(head);
	}
}

void footer(void)
{
	int openCheck;
	FILE *foot;

	openCheck=0;
	if ((foot=fopen(FOOTER,"r"))!=NULL)
		openCheck=1;

	if (openCheck) {
		fprintf(html,"\n<!-- User defined footer begins -->\n");
		while(fgets(cLine,MAXLINE,foot)!=NULL)
			fprintf(html,"%s",cLine);
		fprintf(html,"<!-- User defined footer ends -->\n\n");
		fclose(foot);
	}
}

void htmlhead(void)
{
	int openCheck;
	FILE *head;

	openCheck=0;
	if ((head=fopen(HHEAD,"r"))!=NULL)
		openCheck=1;

	if (openCheck) {
		while(fgets(cLine,MAXLINE,head)!=NULL)
			fprintf(html,"%s",cLine);
		fclose(head);
	}
}

void nick_exclude(void)
{
	FILE *exclude;
	int openCheck, g;
	char poistnick[60];

	openCheck=0;
	if ((exclude=fopen(EXNICK,"r"))!=NULL)
		openCheck=1;

	if (openCheck != 0) {
		while(fscanf(exclude,"%s",poistnick)!=EOF) {
			for (g=0;g<=usersInUse;g++) {
				if ((strncmp(user[g].nick,poistnick,strlen(poistnick)-1)) == 0) {
					zero_nick(g);
					if (debug==2)
						printf("exclude: %s\n",user[g].nick);
					break;
				}
			}
		}
		fclose(exclude);
	}


}

void loadHistory(void)
{

	FILE *db;
	USER tempu;
	SETUP setu;
	SETUPVERSION setuver;
	int openCheck, e, t;

	openCheck=0;
	if ((db=fopen(USERDB,"r"))!=NULL)
		openCheck=1;

	if (openCheck==1) {
		if (!fread(&setuver,sizeof(SETUPVERSION),1,db)) {
			printf("Error while reading version from historyfile!\n");
			exit(1);
		}
		if (setuver.version!=3) {
			printf("Wrong version of historyfile!\n");
			printf("Use dbconvert if the current historyfile is from version 0.16.\n");
			exit(1);
		}
		prevTime=setuver.dummy;
		if (!fread(&setu,sizeof(SETUP),1,db)) {
			printf("Error while reading setup from historyfile!\n");
			exit(1);
		}
		for (e=0;e<=23;e++)
			h[e]=setu.h[e];
		day=setu.day;
		quotenbr=setu.quotenbr;
		totalLines=setu.totalLines;
		continueFromLine=setu.currentLine;
		strncpy(oldlastline,setu.lastline,512);
		topickayt=setu.topickayt;
		for (e=0;e<=4;e++)
			strncpy(topic[e],setu.topic[e],256);

		for (e=0;e<=30;e++) {
			for (t=0;t<=3;t++)
				dayact[e][t]=setu.dayact[e][t];
		}


		while(fread(&tempu,sizeof(USER),1,db)==1) {
			strcpy(user[usersInUse+1].nick,tempu.nick);
			for (e=0;e<=3;e++)
				user[usersInUse+1].talk[e]=tempu.talk[e];
			user[usersInUse+1].join=tempu.join;
			user[usersInUse+1].yell=tempu.yell;
			user[usersInUse+1].ques=tempu.ques;
			user[usersInUse+1].mono1=tempu.mono1;
			user[usersInUse+1].mono2=tempu.mono2;
			user[usersInUse+1].kick=tempu.kick;
			user[usersInUse+1].kicked=tempu.kicked;
			user[usersInUse+1].action=tempu.action;
			user[usersInUse+1].linelength=tempu.linelength;
			user[usersInUse+1].ttalk=tempu.ttalk;
			if (setu.minorver==22) {
				user[usersInUse+1].happy=tempu.happy;
				user[usersInUse+1].sad=tempu.sad;
			} else {
				user[usersInUse+1].happy=0;
				user[usersInUse+1].sad=0;
			}
			strcpy(user[usersInUse+1].quote,tempu.quote);
			usersInUse++;
		}
		fclose(db);
	} else {
		printf("Warning: Unable to open historyfile.\n");
		printf("         If this is the first run then ignore this message.\n\n"); 
	}
}

void saveHistory(void)
{
	FILE *db;
	USER tempu;
	SETUP setu;
	SETUPVERSION setuver;
	int openCheck, t, e;

	openCheck=0;
	if ((db=fopen(USERDB,"w"))!=NULL)
		openCheck=1;

	if (openCheck==1) {
		setuver.version=3;
		setuver.dummy=prevTime;
		fwrite(&setuver,sizeof(SETUPVERSION),1,db);

		for (e=0;e<=23;e++)
			setu.h[e]=h[e];
		setu.day=day;
		setu.quotenbr=quotenbr;
		setu.totalLines=totalLines;
		setu.currentLine=countedLines;
		strncpy(setu.lastline,cLine,512);
		setu.minorver=22;

		for (e=0;e<=4;e++)
			strncpy(setu.topic[e],topic[e],256);
		setu.topickayt=topickayt;

		for (e=0;e<=30;e++) {
			for (t=0;t<=3;t++)
				setu.dayact[e][t]=dayact[e][t];
		}

		fwrite(&setu,sizeof(SETUP),1,db);

		for (t=0;t<=usersInUse;t++) {
			strcpy(tempu.nick,user[t].nick);
			for (e=0;e<=3;e++)
				tempu.talk[e]=user[t].talk[e];
			tempu.join=user[t].join;
			tempu.yell=user[t].yell;
			tempu.ques=user[t].ques;
			tempu.mono1=user[t].mono1;
			tempu.mono2=user[t].mono2;
			tempu.kick=user[t].kick;
			tempu.kicked=user[t].kicked;
			tempu.action=user[t].action;
			tempu.linelength=user[t].linelength;
			tempu.ttalk=user[t].ttalk;
			tempu.happy=user[t].happy;
			tempu.sad=user[t].sad;
			strcpy(tempu.quote,user[t].quote);
			fwrite(&tempu,sizeof(USER),1,db);
		}
		fclose(db);
	} else {
		fprintf(stderr,"\nError:\nUnable to open historydata!\n\n");
		exit(1);
	}

}

void printTopic(void)
{
	char topicmaker[NICKMAX];
	int v, makerpit;


	fprintf(html,"<hr width=\"95%%\">\n");
	if (globalselect[10]!='0')
		getconf("topics:",1);
	else
		getconf("rtopics:",1);
	fprintf(html,"<font color=\"%s\" size=\"4\" face=\"Verdana\"><i>%s</i></font><br>\n",cf1,langtemp);
	fprintf(html,"<table border=\"0\" cellpadding=\"2\" cellspacing=\"2\">\n");
	getconf("change:",1);
	fprintf(html,"<tr><td align=\"left\" bgcolor=\"%s\"><font color=\"%s\" size=\"3\" face=\"Times New Roman\"><i><b> %s </b></i></font></td>\n",ct3,cf3,langtemp);
	getconf("ntopic:",1);
	fprintf(html,"<td align=\"left\" bgcolor=\"%s\"><font color=\"%s\" size=\"3\" face=\"Times New Roman\"><i><b> %s </b></i></font></td></tr>\n",ct3,cf3,langtemp);

	for (v=0;v<=topickayt;v++) {

		sscanf(topic[v],"%s",topicmaker);
		makerpit=strlen(topicmaker);
		removeHtml(topic[v], strlen(topic[v]), 1);

		fprintf(html,"<tr><td align=\"left\" bgcolor=\"%s\"><table border=0 cellspacing=0 cellpadding=0 width=\"100%%\"><tr><td><font color=\"%s\" size=\"2\" face=\"Verdana\">%s</font></td></tr></table></td>\n",ct1,cf3,topicmaker);
		fprintf(html,"<td align=\"left\" bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\"><i>%s</i></font></td>\n",ct1,cf3,topic[v]+makerpit+1);
		fprintf(html,"</tr>\n");
	}

	fprintf(html,"</table>\n");

}

int searchNick(char sNick[NICKMAX], int connect, int from, int casesens)
{
	int g;

	if (casesens == 0) {
		if (usersInUse!=-1) {
			for (g=from;g<=usersInUse;g++) {
				if ((strncasecmp(user[g].nick,sNick,connect)) == 0) {
					return g;
				}
			}
		}
	} else {
		if (usersInUse!=-1) {
			for (g=from;g<=usersInUse;g++) {
				if ((strncmp(user[g].nick,sNick,connect)) == 0) {
					return g;
				}
			}
		}
	}

	return -1;
}

void removeHtml(char html_txt[], int html_len, int isTopic)
{
	int e;
	char buffer[1024];
	char kirjain[2];
	strcpy(buffer,"\0");

	for (e=0;e<=html_len;e++) {
		switch (html_txt[e]) {
			case 60:
				strcat(buffer,"&lt;");
				break;
			case 62:
				strcat(buffer,"&gt;");
				break;
			default:
				kirjain[0]=html_txt[e];
				strncat(buffer,kirjain,1);
				break;
		}
	}
	if (isTopic)
		strncpy(html_txt,buffer,256);
	else
		strncpy(html_txt,buffer,MAXLINE);
}

void nickjoin(void)
{
	char njrivi[MAXLINE], joinnick[NICKMAX], tempnick[NICKMAX];
	int joinpit, nickpit, jnicknbr, tnicknbr, tempnbr, l, nickrivit;
	FILE *nickj;

	if ((nickj=fopen(JNICK,"r"))==NULL) {
                fprintf(stderr,"\nError:\nUnable to open file \"%s\" for reading nickjoins!\n\n",JNICK);
                exit(0);
    }

	while (fgets(njrivi,MAXLINE,nickj)!=NULL) {

		/* check if line begins with # */
		if (njrivi[0]==35)
			continue;

		/* check if line is too short */
		if (strlen(njrivi)<=3)
			continue;

		sscanf(njrivi,"%s",joinnick);
		joinpit=strlen(joinnick);
		nickpit=0;
		nickrivit=0;

		/* check if nick exists, if not then create it */
		jnicknbr=searchNick(joinnick, -1, 0, njcasesensitive);
		if (jnicknbr==-1) {
			if (debug==2)
				printf("nickjoin: created nick %s\n",joinnick);
			strcpy(user[usersInUse+1].nick,joinnick);
			for (l=0;l<=3;l++)
				user[usersInUse+1].talk[l]=0;
			user[usersInUse+1].join=1;
			user[usersInUse+1].yell=0;
			user[usersInUse+1].ques=0;
			user[usersInUse+1].mono1=1;
			user[usersInUse+1].mono2=0;
			user[usersInUse+1].linelength=0.0;
			user[usersInUse+1].ttalk=0;
			user[usersInUse+1].action=0;
			user[usersInUse+1].kick=0;
			user[usersInUse+1].kicked=0;
			user[usersInUse+1].happy=0;
			user[usersInUse+1].sad=0;
			strcpy(user[usersInUse+1].quote,"No quote");
			usersInUse++;
			jnicknbr=usersInUse;
		}

		nickrivit=user[jnicknbr].ttalk;

		while (joinpit+nickpit<=strlen(njrivi)-2) {
			sscanf(njrivi+joinpit+nickpit,"%s",tempnick);

			/* if last char is a star */
			if (tempnick[strlen(tempnick)-1]==42) {
				tnicknbr=0;
				tempnbr=0;
				while (tnicknbr!=-1) {
					tnicknbr=searchNick(tempnick, strlen(tempnick)-1, tempnbr, njcasesensitive);

					if (jnicknbr==tnicknbr) {
						tempnbr++;
						continue;
					}
					if (tnicknbr!=-1) {
						/* copy to right one */
						if (debug==2)
							printf("nickjoin(*): %s -> %s\n",user[tnicknbr].nick,user[jnicknbr].nick);
						user[jnicknbr].talk[0]+=user[tnicknbr].talk[0];
						user[jnicknbr].talk[1]+=user[tnicknbr].talk[1];
						user[jnicknbr].talk[2]+=user[tnicknbr].talk[2];
						user[jnicknbr].talk[3]+=user[tnicknbr].talk[3];
						user[jnicknbr].join+=user[tnicknbr].join;
						if (!user[jnicknbr].linelength)
							user[jnicknbr].linelength=user[tnicknbr].linelength;
						else
							if (user[tnicknbr].linelength)
								user[jnicknbr].linelength=(user[jnicknbr].ttalk*user[jnicknbr].linelength+user[tnicknbr].ttalk*user[tnicknbr].linelength)/((float)user[jnicknbr].ttalk+(float)user[tnicknbr].ttalk);
						user[jnicknbr].yell+=user[tnicknbr].yell;
						user[jnicknbr].ques+=user[tnicknbr].ques;
						user[jnicknbr].mono2+=user[tnicknbr].mono2;
						user[jnicknbr].action+=user[tnicknbr].action;
						user[jnicknbr].kick+=user[tnicknbr].kick;
						user[jnicknbr].kicked+=user[tnicknbr].kicked;
						user[jnicknbr].ttalk+=user[tnicknbr].ttalk;
						user[jnicknbr].happy+=user[tnicknbr].happy;
						user[jnicknbr].sad+=user[tnicknbr].sad;
						if (user[tnicknbr].ttalk>=nickrivit) {
							strcpy(user[jnicknbr].quote,user[tnicknbr].quote);
							nickrivit=user[tnicknbr].ttalk;
						}

						/* clear the old nick */
						zero_nick(tnicknbr);

						/* move to next */
						tempnbr=tnicknbr+1;
					}
					if (tempnbr>usersInUse) {
						break;
					}
				}
			} else
				tnicknbr=searchNick(tempnick, -1, 0, njcasesensitive);


			/* if nick isn't found */
			if (tnicknbr==-1) {
				if (debug==2)
					printf("nickjoin: nick %s not found.\n",tempnick);
				nickpit=nickpit+strlen(tempnick)+1;
				continue;
			} else {
				/* copy to right one */
				if (debug==2)
					printf("nickjoin: %s -> %s\n",user[tnicknbr].nick,user[jnicknbr].nick);
				user[jnicknbr].talk[0]+=user[tnicknbr].talk[0];
				user[jnicknbr].talk[1]+=user[tnicknbr].talk[1];
				user[jnicknbr].talk[2]+=user[tnicknbr].talk[2];
				user[jnicknbr].talk[3]+=user[tnicknbr].talk[3];
				user[jnicknbr].join+=user[tnicknbr].join;
				if(!user[jnicknbr].linelength)
					user[jnicknbr].linelength=user[tnicknbr].linelength;
				else
					user[jnicknbr].linelength=(user[jnicknbr].ttalk*user[jnicknbr].linelength+user[tnicknbr].ttalk*user[tnicknbr].linelength)/((float)user[jnicknbr].ttalk+(float)user[tnicknbr].ttalk);
				user[jnicknbr].yell+=user[tnicknbr].yell;
				user[jnicknbr].action+=user[tnicknbr].action;
				user[jnicknbr].kick+=user[tnicknbr].kick;
				user[jnicknbr].kicked+=user[tnicknbr].kicked;
				user[jnicknbr].ques+=user[tnicknbr].ques;
				user[jnicknbr].mono2+=user[tnicknbr].mono2;
				user[jnicknbr].ttalk+=user[tnicknbr].ttalk;
				user[jnicknbr].happy+=user[tnicknbr].happy;
				user[jnicknbr].sad+=user[tnicknbr].sad;
				if (user[tnicknbr].ttalk>=nickrivit) {
					strcpy(user[jnicknbr].quote,user[tnicknbr].quote);
					nickrivit=user[tnicknbr].ttalk;
				}

				/* clear the old nick */
				zero_nick(tnicknbr);
			}
			nickpit=nickpit+strlen(tempnick)+1;
		}
	}

	fclose(nickj);
}

int comparedTtalk(const void* _a, const void* _b)
{
	const USER* a = (const USER*) _a;
	const USER* b = (const USER*) _b;

	if (a->ttalk > b->ttalk)
		return -1;
	else
		if (a->ttalk < b->ttalk)
			return 1;
	return 0;
}

void sort_nick(void)
{
	if (usersInUse>0)
		qsort((void *)user,usersInUse+1,sizeof(user[0]),comparedTtalk);
}

void zero_nick(int znick)
{
	user[znick].talk[0]=0;
	user[znick].talk[1]=0;
	user[znick].talk[2]=0;
	user[znick].talk[3]=0;
	strcpy(user[znick].quote,"No quote");
	user[znick].join=1;
	user[znick].linelength=0.0;
	user[znick].yell=0;
	user[znick].ques=0;
	user[znick].mono1=0;
	user[znick].mono2=0;
	user[znick].ttalk=0;
	user[znick].place=0;
	user[znick].dummy=0;
	user[znick].action=0;
	user[znick].kick=0;
	user[znick].kicked=0;
	user[znick].opped=0;
	user[znick].deopped=0;
	user[znick].happy=0;
	user[znick].sad=0;
	user[znick].foo1=0;
}

void dayactrotate(void)
{
	int j, k;

	for (j=29;j>=0;j--) {
		for (k=0;k<=3;k++) {
			dayact[j+1][k]=dayact[j][k];
		}
	}

	for (k=0;k<=3;k++) {
		dayact[0][k]=0;
	}
}


void gapfiller(char inputtxt[MAXLINE], char keyword[64])
{
	char buffer[128];
	char kirjain[3];
	int pit, paikka;

	pit=0;
	strcpy(langtemp,"\0");

	while (sscanf(inputtxt+pit,"%s",buffer)!=EOF) {

		pit+=strlen(buffer);

		if ((buffer[0]=='%') && (isdigit(buffer[1]))) {
			strncpy(kirjain,buffer+1,2);
			paikka=atoi(kirjain);
			if (paikka>=11) {
				fprintf(stderr,"\nError:\nThe variable number in the config for line \"%s\" is something greater than 10.\n\n",keyword); 
				exit(1);
			}
			if (strlen(langtemp)!=0)
				strcat(langtemp," ");
			strcat(langtemp,fill[paikka]);

			if (strlen(buffer)>2)
				strcat(langtemp,buffer+2);
			pit++;
		} else {
			if (strlen(langtemp)!=0)
				strcat(langtemp," ");
			strcat(langtemp,buffer);
			pit++;
		}
		if (pit>=strlen(inputtxt))
			break;
	}

}

void getconf(char keyword[64], int fillgaps)
{
	int check;
	char buffer[128];
	char templine[MAXLINE];

	check=0;

	rewind(configfile);
	strcpy(buffer,"\0");

	while (fgets(confline,MAXLINE,configfile)!=NULL) {

		sscanf(confline,"%s",buffer);

		if ((confline[0]=='/') && (confline[1]=='/')) {
			strcpy(buffer,"\0");
			continue;
		}
		if (strcmp(buffer,keyword)==0) {
			check=1;
			if (fillgaps)
				gapfiller(confline+strlen(keyword)+1, keyword);
			else
				confline[strlen(confline)-1]='\0';
			strcpy(templine,confline+strlen(keyword)+1);
			strcpy(confline, templine);

			/* convert true and false to 1 and 0 */
			if (strcasecmp(confline,"true")==0)
				strcpy(confline,"1");
			if ((strcasecmp(confline,"false")==0) || (strcasecmp(confline,"none")==0))
				strcpy(confline,"0");

			break;
		}
		strcpy(buffer,"\0");
	}
	if (check==0) {
		strcpy(buffer,keyword);
		buffer[strlen(buffer)-1]='\0';
		fprintf(stderr,"\nError:\nConfig for \"%s\" is missing from the file \"%s\".\n\n",buffer, configInUse);
		exit(1);
	}
}

void timeofday(void)
{
	int j, night[10], early[10], after[10], even[10], k, l;

	for (j=0;j<=9;j++) {
		night[j]=-1;
		early[j]=-1;
		after[j]=-1;
		even[j]=-1;
	}

	for (j=0;j<=usersInUse;j++) {
		/* hours 0-6 */
		for (k=0;k<=9;k++) {
			if ((night[k]==-1) || (user[j].talk[0]>user[(night[k])].talk[0])) {
				for (l=9;l>=k+1;l--)
					night[l]=night[l-1];
				night[k]=j;
				break;
			}
		}
		/* hours 6-12 */
		for (k=0;k<=9;k++) {
			if ((early[k]==-1) || (user[j].talk[1]>user[(early[k])].talk[1])) {
				for (l=9;l>=k+1;l--)
					early[l]=early[l-1];
				early[k]=j;
				break;
			}
		}
		/* hours 12-18 */
		for (k=0;k<=9;k++) {
			if ((after[k]==-1) || (user[j].talk[2]>user[(after[k])].talk[2])) {
				for (l=9;l>=k+1;l--)
					after[l]=after[l-1];
				after[k]=j;
				break;
			}
		}
		/* hours 18-24 */
		for (k=0;k<=9;k++) {
			if ((even[k]==-1) || (user[j].talk[3]>user[(even[k])].talk[3])) {
				for (l=9;l>=k+1;l--)
					even[l]=even[l-1];
				even[k]=j;
				break;
			}
		}
	}

	fprintf(html,"<hr width=\"95%%\">\n");
	getconf("timeofday:",1);
	fprintf(html,"<font color=\"%s\" size=\"4\" face=\"Verdana\"><i>%s</i></font><br>\n",cf1,langtemp);
	fprintf(html,"<table border=\"0\" cellpadding=\"2\" cellspacing=\"2\">\n");
	fprintf(html,"<tr><td>&nbsp;</td>\n");

	fprintf(html,"<td align=center bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct3,cf3);
	getconf("tod0-6:",1);
	fprintf(html,"<i><b>%s</b></i></font><br><font color=\"%s\" size=\"1\" face=\"Verdana\">",langtemp,cf3);
	getconf("hours0-6:",1);
	fprintf(html,"(%s)</font></td>\n",langtemp);

	fprintf(html,"<td align=center bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct3,cf3);
	getconf("tod6-12:",1);
	fprintf(html,"<i><b>%s</b></i></font><br><font color=\"%s\" size=\"1\" face=\"Verdana\">",langtemp,cf3);
	getconf("hours6-12:",1);
	fprintf(html,"(%s)</font></td>\n",langtemp);

	fprintf(html,"<td align=center bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct3,cf3);
	getconf("tod12-18:",1);
	fprintf(html,"<i><b>%s</b></i></font><br><font color=\"%s\" size=\"1\" face=\"Verdana\">",langtemp,cf3);
	getconf("hours12-18:",1);
	fprintf(html,"(%s)</font></td>\n",langtemp);

	fprintf(html,"<td align=center bgcolor=\"%s\"><font color=\"%s\" size=\"2\" face=\"Verdana\">",ct3,cf3);
	getconf("tod18-24:",1);
	fprintf(html,"<i><b>%s</b></i></font><br><font color=\"%s\" size=\"1\" face=\"Verdana\">",langtemp,cf3);
	getconf("hours18-24:",1);
	fprintf(html,"(%s)</font></td>\n",langtemp);

	for (j=0;j<=9;j++) {

		if (!user[(night[j])].talk[0] && !user[(early[j])].talk[1] && !user[(after[j])].talk[2] && !user[(even[j])].talk[3])
			break;

		fprintf(html,"<tr><td align=\"right\" bgcolor=\"%s\">",ct4);
		fprintf(html,"<font color=\"%s\" size=\"1\" face=\"Verdana\">%d</font>",cf4,j+1);

		if (user[(night[j])].talk[0]) {
			fprintf(html,"</td><td align=\"left\" valign=\"middle\" bgcolor=\"%s\">",ct5);
			fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\">%s - %d</font></td>",cf4, user[(night[j])].nick, user[(night[j])].talk[0]);
		} else {
			fprintf(html,"</td><td align=\"left\" valign=\"middle\" bgcolor=\"%s\">",ct5);
			fprintf(html,"&nbsp;</td>");
		}

		if (user[(early[j])].talk[1]) {
			fprintf(html,"<td align=\"left\" valign=\"middle\" bgcolor=\"%s\">",ct5);
			fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\">%s - %d</font></td>\n",cf4, user[(early[j])].nick, user[(early[j])].talk[1]);
		} else {
			fprintf(html,"<td align=\"left\" valign=\"middle\" bgcolor=\"%s\">",ct5);
			fprintf(html,"&nbsp;</td>\n");
		}

		if (user[(after[j])].talk[2]) {
			fprintf(html,"<td align=\"left\" valign=\"middle\" bgcolor=\"%s\">",ct5);
			fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\">%s - %d</font></td>",cf4, user[(after[j])].nick, user[(after[j])].talk[2]);
		} else {
			fprintf(html,"<td align=\"left\" valign=\"middle\" bgcolor=\"%s\">",ct5);
			fprintf(html,"&nbsp;</td>");
		}

		if (user[(even[j])].talk[3]) {
			fprintf(html,"<td align=\"left\" valign=\"middle\" bgcolor=\"%s\">",ct5);
			fprintf(html,"<font color=\"%s\" size=\"2\" face=\"Verdana\">%s - %d</font></td></tr>\n",cf4, user[(even[j])].nick, user[(even[j])].talk[3]);
		} else {
			fprintf(html,"<td align=\"left\" valign=\"middle\" bgcolor=\"%s\">",ct5);
			fprintf(html,"&nbsp;</td></tr>\n");
		}
	}

	fprintf(html,"</table><br>\n");

}

int timetodate(int daycorrection, int isShort)
{
	char tempdate[128], dateweekday[5], dateday[5], datemonth[5], datetime[10], dateyear[5];
	time_t currentdate;

	currentdate=time(NULL);
	currentdate=currentdate+(3600*timezoneCorrection)-(86400*daycorrection);

	sprintf(tempdate,"%s",asctime(localtime(&currentdate)));
	sscanf(tempdate,"%s %s %s %s %s",dateweekday,datemonth,dateday,datetime,dateyear);

	if(isShort) {
		if (strcmp("Jan",datemonth)==0)
			sprintf(datestamp,"%d.1.\n",atoi(dateday));
		else if (strcmp("Feb",datemonth)==0)
			sprintf(datestamp,"%d.2.\n",atoi(dateday));
		else if (strcmp("Mar",datemonth)==0)
			sprintf(datestamp,"%d.3.\n",atoi(dateday));
		else if (strcmp("Apr",datemonth)==0)
			sprintf(datestamp,"%d.4.\n",atoi(dateday));
		else if (strcmp("May",datemonth)==0)
			sprintf(datestamp,"%d.5.\n",atoi(dateday));
		else if (strcmp("Jun",datemonth)==0)
			sprintf(datestamp,"%d.6.\n",atoi(dateday));
		else if (strcmp("Jul",datemonth)==0)
			sprintf(datestamp,"%d.7.\n",atoi(dateday));
		else if (strcmp("Aug",datemonth)==0)
			sprintf(datestamp,"%d.8.\n",atoi(dateday));
		else if (strcmp("Sep",datemonth)==0)
			sprintf(datestamp,"%d.9.\n",atoi(dateday));
		else if (strcmp("Oct",datemonth)==0)
			sprintf(datestamp,"%d.10.\n",atoi(dateday));
		else if (strcmp("Nov",datemonth)==0)
			sprintf(datestamp,"%d.11.\n",atoi(dateday));
		else if (strcmp("Dec",datemonth)==0)
			sprintf(datestamp,"%d.12.\n",atoi(dateday));
	} else {
		if (strcmp("Jan",datemonth)==0)
			sprintf(datestamp,"%s %d.1.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
		else if (strcmp("Feb",datemonth)==0)
			sprintf(datestamp,"%s %d.2.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
		else if (strcmp("Mar",datemonth)==0)
			sprintf(datestamp,"%s %d.3.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
		else if (strcmp("Apr",datemonth)==0)
			sprintf(datestamp,"%s %d.4.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
		else if (strcmp("May",datemonth)==0)
			sprintf(datestamp,"%s %d.5.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
		else if (strcmp("Jun",datemonth)==0)
			sprintf(datestamp,"%s %d.6.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
		else if (strcmp("Jul",datemonth)==0)
			sprintf(datestamp,"%s %d.7.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
		else if (strcmp("Aug",datemonth)==0)
			sprintf(datestamp,"%s %d.8.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
		else if (strcmp("Sep",datemonth)==0)
			sprintf(datestamp,"%s %d.9.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
		else if (strcmp("Oct",datemonth)==0)
			sprintf(datestamp,"%s %d.10.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
		else if (strcmp("Nov",datemonth)==0)
			sprintf(datestamp,"%s %d.11.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
		else if (strcmp("Dec",datemonth)==0)
			sprintf(datestamp,"%s %d.12.%d\n",dateweekday,atoi(dateday),atoi(dateyear));
	}

	if ((strcmp("Sun",dateweekday)==0) || (strcmp("Sat",dateweekday)==0))
		return 1;
	else
		return 0;
}
